#!/usr/bin/env bash

# GooseCode Linux Desktop Integration Installer
# Run this script from the extracted goosecode directory to set up:
# - Desktop entry (app launcher)
# - Application icon
# - MIME type associations (.goosecode files)

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
APP_ID="com.gooseclip.goosecode"
APP_NAME="goosecode"

# Detect if we're in the source tree or the release bundle
if [[ -f "$SCRIPT_DIR/goosecode" ]]; then
  # Release bundle - goosecode executable is in the same directory
  BUNDLE_DIR="$SCRIPT_DIR"
  ICON_SOURCE="$BUNDLE_DIR/data/$APP_ID.png"
  DESKTOP_SOURCE="$BUNDLE_DIR/data/$APP_ID.desktop"
  MIME_SOURCE="$BUNDLE_DIR/data/$APP_ID.mime.xml"
elif [[ -f "$SCRIPT_DIR/../goosecode.png" ]]; then
  # Source tree - linux/install.sh
  BUNDLE_DIR=""
  ICON_SOURCE="$SCRIPT_DIR/goosecode.png"
  DESKTOP_SOURCE="$SCRIPT_DIR/$APP_ID.desktop"
  MIME_SOURCE="$SCRIPT_DIR/$APP_ID.mime.xml"
else
  echo "Error: Cannot find GooseCode files."
  echo "Run this script from the goosecode directory after extracting the tarball."
  exit 1
fi

# XDG directories
ICONS_DIR="${XDG_DATA_HOME:-$HOME/.local/share}/icons/hicolor"
APPS_DIR="${XDG_DATA_HOME:-$HOME/.local/share}/applications"
MIME_DIR="${XDG_DATA_HOME:-$HOME/.local/share}/mime/packages"

echo "=== GooseCode Desktop Integration Installer ==="
echo ""

# Check for required files
if [[ ! -f "$ICON_SOURCE" ]]; then
  echo "Error: Icon not found at $ICON_SOURCE"
  exit 1
fi

echo "Installing for user: $USER"
echo "Icon source: $ICON_SOURCE"
echo ""

# Create directories
mkdir -p "$ICONS_DIR/256x256/apps"
mkdir -p "$ICONS_DIR/128x128/apps"
mkdir -p "$ICONS_DIR/64x64/apps"
mkdir -p "$ICONS_DIR/48x48/apps"
mkdir -p "$APPS_DIR"
mkdir -p "$MIME_DIR"

# Install icon (copy to multiple sizes for better scaling)
echo "Installing icon..."
cp "$ICON_SOURCE" "$ICONS_DIR/256x256/apps/$APP_ID.png"
# Create smaller versions if ImageMagick is available
if command -v convert &> /dev/null; then
  convert "$ICON_SOURCE" -resize 128x128 "$ICONS_DIR/128x128/apps/$APP_ID.png"
  convert "$ICON_SOURCE" -resize 64x64 "$ICONS_DIR/64x64/apps/$APP_ID.png"
  convert "$ICON_SOURCE" -resize 48x48 "$ICONS_DIR/48x48/apps/$APP_ID.png"
  echo "  ✓ Created multiple icon sizes"
else
  # Fall back to just using the 256x256 version everywhere
  cp "$ICON_SOURCE" "$ICONS_DIR/128x128/apps/$APP_ID.png"
  cp "$ICON_SOURCE" "$ICONS_DIR/64x64/apps/$APP_ID.png"
  cp "$ICON_SOURCE" "$ICONS_DIR/48x48/apps/$APP_ID.png"
  echo "  ✓ Installed icon (install imagemagick for better scaling)"
fi

# Create desktop file with absolute path to executable
echo "Installing desktop entry..."
if [[ -n "$BUNDLE_DIR" ]]; then
  EXEC_PATH="$BUNDLE_DIR/goosecode"
  cat > "$APPS_DIR/$APP_ID.desktop" << EOF
[Desktop Entry]
Name=GooseCode
Comment=GooseCode Application
Exec="$EXEC_PATH" %U
Icon=$APP_ID
Terminal=false
Type=Application
Categories=Development;IDE;
StartupWMClass=$APP_ID
MimeType=x-scheme-handler/goosecode;application/x-goosecode;
EOF
  echo "  ✓ Created desktop entry pointing to: $EXEC_PATH"
else
  # Source tree - use generic name (assumes goosecode is in PATH)
  cp "$DESKTOP_SOURCE" "$APPS_DIR/$APP_ID.desktop"
  echo "  ✓ Installed desktop entry"
fi

# Install MIME type if available
if [[ -f "$MIME_SOURCE" ]]; then
  echo "Installing MIME type..."
  cp "$MIME_SOURCE" "$MIME_DIR/$APP_ID.xml"
  echo "  ✓ Installed MIME type for .goosecode files"
fi

# Update system databases
echo "Updating system databases..."

# Update icon cache
if command -v gtk-update-icon-cache &> /dev/null; then
  gtk-update-icon-cache -f -t "$ICONS_DIR" 2>/dev/null || true
  echo "  ✓ Updated icon cache"
fi

# Update desktop database
if command -v update-desktop-database &> /dev/null; then
  update-desktop-database "$APPS_DIR" 2>/dev/null || true
  echo "  ✓ Updated desktop database"
fi

# Update MIME database
if command -v update-mime-database &> /dev/null; then
  update-mime-database "${XDG_DATA_HOME:-$HOME/.local/share}/mime" 2>/dev/null || true
  echo "  ✓ Updated MIME database"
fi

echo ""
echo "=== Installation Complete ==="
echo ""
echo "GooseCode should now appear in your application launcher."
echo "You may need to log out and back in for all changes to take effect."
echo ""
echo "To uninstall, run:"
echo "  rm ~/.local/share/applications/$APP_ID.desktop"
echo "  rm ~/.local/share/icons/hicolor/*/apps/$APP_ID.png"
echo "  rm ~/.local/share/mime/packages/$APP_ID.xml"
